<?php

namespace App\Filament\Resources;

use App\Enums\Roles;
use App\Enums\Status;
use App\Filament\Resources\BlogResource\Pages;
use App\Filament\Resources\BlogResource\RelationManagers;
use App\Forms\Components\LarabergEditor;
use App\Library\Functions;
use App\Models\Blog;
use App\Models\Category;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class BlogResource extends Resource
{
    protected static ?string $model = Blog::class;

    protected static ?string $navigationIcon = 'heroicon-o-bookmark-square';

    static function getOptions(){
        $user = auth()->user();

        $options = [
            Status::DRAFT->value => Status::DRAFT->value,
            Status::PENDING->value => Status::PENDING->value,
        ];

        if($user->is_editor) {
            $options = array_merge($options, [
                Status::PUBLISHED->value => Status::PUBLISHED->value,
                Status::UNPUBLISHED->value => Status::UNPUBLISHED->value,
                Status::SCHEDULED->value => Status::SCHEDULED->value,
            ]);
        }

        return $options;
    }

    public static function form(Form $form): Form {
        $user = auth()->user();
        return $form->schema([
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255)
                    ->live()
                    ->afterStateUpdated(function (Get $get, Set $set, ?string $old, ?string $state) {
                         if (($get('slug') ?? '') !== str($old)->slug()) return;
                        $set('slug', str($state)->slug());
                    }),
                Forms\Components\TextInput::make('slug')
                    ->required()
                    ->unique(column: 'slug', ignoreRecord: true)
                    ->maxLength(255),
                Forms\Components\Select::make('category_id')
                    ->searchable()
                    ->required()
                    ->label('Category')
                    ->relationship('category')
                    ->options(Category::isChild()->get()->pluck('name', 'id')),
                Select::make('author_id')
                    ->required()
                    ->label('Author')
                    ->options($user->is_admin ? User::isWriter()->get()->pluck('name_role', 'id') : [$user->id => $user->name_role])
                    ->default(auth()->id()),
                Checkbox::make('is_featured')
                    ->label('Set as Featured Post'),
                Section::make('Content')
                    ->description((new Functions)->context())
                    ->schema([
                        LarabergEditor::make('content')
                            ->hiddenLabel(true)
                            ->required()
                    ]),
                Forms\Components\FileUpload::make('image')
                    ->required()
                    ->preserveFilenames()
                    ->image()
                    ->imageEditor()
                    // ->visible()
                    ->visibility('public')
                    ->optimize('avif')
                    ->label('Featured Image'),
                Forms\Components\TagsInput::make('tags')
                    ->required()
                    ->placeholder("Add Keyword and Click enter")
                    ->columnSpanFull(),
                Forms\Components\Textarea::make('description')
                    ->required()
                    ->maxLength(65535)
                    ->columnSpanFull(),
                DateTimePicker::make('published_at')
                    ->label('Publishing Date')
                    ->placeholder('Publish Date')
                    ->default(now())
                    ->closeOnDateSelection(),
                Select::make('status')
                    ->required()
                    ->options(self::getOptions())
            ]);
    }

    public static function table(Table $table): Table {
        return $table->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable(),
                Tables\Columns\TextColumn::make('category.name')
                    ->label('Category')
                    ->searchable(),
                Tables\Columns\TextColumn::make('author_id')
                    ->state(function (Blog $record) {
                        return $record->author?->fullname;
                    })
                    ->label('Author')
                    ->searchable(),
                Tables\Columns\TextColumn::make('published_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }



    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBlogs::route('/'),
            'create' => Pages\CreateBlog::route('/create'),
            'edit' => Pages\EditBlog::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->latest()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
