<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use App\Enums\Roles;
use App\Enums\Status;
use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasName;
use Filament\Panel;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable implements FilamentUser, HasName {
    use HasApiTokens, HasFactory, Notifiable, HasUuids;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [ 'firstname', 'lastname', 'role', 'email', 'password', 'status', 'bio', 'facebook', 'linkedin', 'twitter', 'instagram' ];

    public function getFilamentName(): string {
        return "{$this->firstname} {$this->lastname}";
    }

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $attributes = [
        'status' => Status::ACTIVE,
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'status' => Status::class,
        'role' => Roles::class
    ];

    function scopeIsSuperAdmin(Builder $query){
        $query->where('role', Roles::SUPERADMIN);
    }

    function scopeIsAdmin(Builder $query){
        $query->where('role', Roles::SUPERADMIN)->orWhere('role', Roles::ADMIN);
    }

    function scopeIsWriter(Builder $query){
        $editors = [Roles::ADMIN, Roles::SUPERADMIN, Roles::EDITOR, Roles::WRITER];
        $query->whereIn('role', $editors);
    }

    function scopeIsEditor(Builder $query){
        $editors = [Roles::ADMIN, Roles::SUPERADMIN, Roles::EDITOR];
        $query->whereIn('role', $editors);
    }

    function getIsEditorAttribute(){
        $editors = [Roles::ADMIN, Roles::SUPERADMIN, Roles::EDITOR];
        return in_array($this->role, $editors);
    }

    function getIsWriterAttribute(){
        $writers = [Roles::ADMIN, Roles::SUPERADMIN, Roles::EDITOR, Roles::WRITER];
        return in_array($this->role, $writers);
    }

    function getIsAdminAttribute(){
        $editors = [Roles::ADMIN, Roles::SUPERADMIN];
        return in_array($this->role, $editors);
    }

    function getFullNameAttribute(){
        return implode(' ', [$this->firstname, $this->lastname]);
    }

    function getNameRoleAttribute(){
        return implode(' - ', [implode(' ', [$this->firstname, $this->lastname]), $this->role->value]);
    }

    function getIsSuperAdminAttribute(){
        $editors = [Roles::SUPERADMIN];
        return in_array($this->role, $editors);
    }

    public function canAccessPanel(Panel $panel): bool {
        $editors = [Roles::ADMIN, Roles::SUPERADMIN, Roles::EDITOR, Roles::WRITER];
        return in_array($this->role, $editors);
    }
}
