<?php

namespace App\Livewire;

use Livewire\Attributes\Layout;
use App\Models\Currency;
use Livewire\Attributes\Url;
use App\Services\ExchangeService;
use Livewire\Component;


#[Layout('layouts.guest', [
    'title' => "Convert Currency"
])]
class Converter extends Component
{
    #[Url]
    public $from = "USD";

    #[Url]
    public $to;

    #[Url]
    public $amount = 1;

    private $steps = [1, 20, 50, 100, 200, 500, 1000, 2000, 5000, 10000];

    public $baseToTarget = [], $targetToBase = [];

    public Currency | null $fromCurrency = null;
    public Currency | null $toCurrency = null;

    public $currencies;

    public $result = null;
    public $rate = null;

    public $base, $target;

    public $rates;
    public $displayRates = [];

    function mount(){
        $this->currencies = Currency::isActive()->get();
        $this->to = session('user_currency') ?? "GBP";

        $this->fromCurrency = Currency::where('symbol', $this->from)->first();
        $this->toCurrency = Currency::where('symbol', $this->to)->first();

        if($this->amount && $this->from && $this->to) $this->exchange();
    }

    function setRates(){
        $this->base = $this->fromCurrency;
        $this->target = $this->toCurrency;
        $this->rate = $this->rate;

        $steps = collect($this->steps);

        $this->baseToTarget = $steps->map(fn($item) => [
            'step' => $item,
            'amount' => $item * $this->rate
        ])->toArray();

        $this->targetToBase = $steps->map(fn($item) => [
            'step' => $item,
            'amount' => $item / $this->rate
        ])->toArray();
    }

    public function percentageDifference($oldValue, $newValue) {
        $difference = $newValue - $oldValue;
        $percentageDifference = ($difference / $oldValue) * 100;

        return $percentageDifference;
    }

    function latestRates($from){
        $currencies = $this->currencies->whereIn('symbol', ['USD', 'EUR', 'GBP', 'YEN', 'CAD', 'CNY'])->where('symbol', '!=', $from);
        $exchangeService = new ExchangeService();
        $today = now();
        $yesterday = now()->subDay();

        [$status, $todayRates] = $exchangeService->date($today, $from, $currencies->implode('symbol', ','));
        if(!$status) return;

        [$status, $yesterdayRates] = $exchangeService->date($yesterday, $from, $currencies->implode('symbol', ','));

        if(!$status) return;

        $this->rates = collect($todayRates)->map(function($rate, $index) use($yesterdayRates) {
            return [
                'today' => $rate,
                'yesterday' => $yesterdayRates[$index],
                'percent' => $this->percentageDifference($rate, $yesterdayRates[$index])
            ];
        });

        $this->displayRates = $this->rates->filter(fn($val, $key) => in_array($key, $currencies->pluck('symbol')->toArray()));
    }

    function swap(){
        $from = $this->from;
        $to = $this->to;
        $this->to = $from;
        $this->from = $to;

        if($this->amount) $this->exchange();
    }

    function exchange(){
        $this->validate([
            'from' => 'required|string',
            'to' => 'required|string',
            'amount' => 'required|numeric'
        ]);

        $this->fromCurrency = Currency::where('symbol', $this->from)->first();
        $this->toCurrency = Currency::where('symbol', $this->to)->first();

        $exchangeService = app(ExchangeService::class);
        [$status, $data] = $exchangeService->convert($this->from, $this->to, $this->amount);

        if($status) {
            $this->result = $data['result'];
            $this->rate = $data['info']['rate'];
        }

        $this->latestRates($this->from);
        $this->setRates();
    }

    public function updated($name){
        $this->exchange();
    }


    public function render()
    {
        return view('livewire.converter')->title($this->amount . " ".$this->from."  to ". $this->to. " - " . $this->fromCurrency->name . " to ". $this->toCurrency->name);
    }
}
